#ifndef _CSockSrvr_CPP
#define _CSockSrvr_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#define WIN32_LEAN_AND_MEAN
#define _WIN32_WINNT 0x0500

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <WinSock2.H>
#include <WinSock.H>
#include <Stdio.H>
#include <Stdlib.H>

#include "CSockSrvr.H"

//NON-STANDARD DEFINITIONS (BEGIN)

#include "../Source/Routines.H"

CSockSrvr gServer;

//NON-STANDARD DEFINITIONS (END)

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void FatalError(char *sFile, int iLine, char *sErrorMsg)
{
	#ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> FatalError.\n");
    #endif

	char sMsg[2048];
	sprintf(sMsg, "CSockSrvr::FatalError-> Error in '%s'\r\nLine: %d\r\nMessage: %s", sFile, iLine, sErrorMsg);
	giErrorCount++;
	WriteSysLog(sMsg);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients DropBadPacketData flag.
*/
void CSockSrvr::DropBadPacketData(int iClient, bool bDropBad)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::DropBadPacketData.\n");
    #endif

    bcDropBadPackets[iClient] = bDropBad;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set icNextClientID to a new value. 
*/
void CSockSrvr::SetNextClientID(int iClientID)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SetNextClientID.\n");
    #endif

    icNextClientID = iClientID;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to get the value of icNextClientID. 
*/
int CSockSrvr::GetNextClientID(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::GetNextClientID.\n");
    #endif

    return icNextClientID;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to get the value of icMaxClients. 
*/
int CSockSrvr::GetMaxClients(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::GetMaxClients.\n");
    #endif

    return icMaxClients;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients bcRawDataMode flag.
*/
void CSockSrvr::SetRawStreamMode(int iClient, bool bRawMode)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SetRawStreamMode.\n");
    #endif

    bcRawDataMode[iClient] = bRawMode;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to get the count of currently connected clients.
*/
int CSockSrvr::GetCurrentClients(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::GetCurrentClients.\n");
    #endif

    return icCurrentClients;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function returns TRUE if the client has data in there receive buffer.

	NOTE: A TRUE return value does not mean that the data is a full packet,
	therfore a call to GetClientData or GetNextClientData may not return data.
*/
bool CSockSrvr::IsClientDataWaiting(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::IsClientDataWaiting.\n");
    #endif

    // Is data waiting to be received
    return (icRecvBufSz[iClient] > 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function returns TRUE if the client has data waiting to be sent.
*/
bool CSockSrvr::IsClientDataPending(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::IsClientDataPending.\n");
    #endif

    // Is data waiting to be sent?
    return (icSendBufSz[iClient] > 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will return TRUE when the client has a empty send buffer.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
*/
bool CSockSrvr::WaitOnClientDataToBeSent(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::WaitonClientDataToBeSent.\n");
    #endif

    bool bRetVal = false;

    while((bRetVal = IsClientDataPending(iClient)) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

    return !bRetVal;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will return TRUE when the client has data in there receive buffer.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
	
	NOTE: A TRUE return value does not mean that the data is a full packet,
	therfore a call to GetClientData or GetNextClientData may not return data.
*/
bool CSockSrvr::WaitOnClientDataToBeRecvd(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::WaitonClientDataToBeRecvd.\n");
    #endif

    bool bRetVal = false;

    while(!(bRetVal = IsClientDataWaiting(iClient)) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

    return bRetVal;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will return TRUE when the client has data in there receive buffer
	and the data was a full packet.

	Returns FALSE if the client has been disconnected, is pending a disconnect or the
	data received is not a full packet.
*/

bool CSockSrvr::GetNextClientData(int iClient, char *outsBuf, int *outiSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::GetNextClientData.\n");
    #endif

    while(!IsClientDataWaiting(iClient) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

    return GetClientData(iClient, outsBuf, outiSize);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to retrieve a clients data from there receive buffer.

	This function returns TRUE if the clients has data to be received,
	the the CriticalSection is open, the data is a full (and valid) packet.
	Otherwise the function returns FALSE.
*/
bool CSockSrvr::GetClientData(int iClient, char *outsBuf, int *outiSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::GetClientData.\n");
    #endif

    if(IsClientDataWaiting(iClient))
    {
        if(bcRawDataMode[iClient])
        {
            if(!TryEnterCriticalSection(&CriticalSection1))
                return false;

            if(bcConnected[iClient] && !bcDisconnect[iClient])
            {
                memcpy(outsBuf, scRecvBuf[iClient], icRecvBufSz[iClient]);
                *outiSize = icRecvBufSz[iClient];
            }
			else{
				LeaveCriticalSection(&CriticalSection1);
				return false;
			}

            LeaveCriticalSection(&CriticalSection1);

            icRecvBufSz[iClient] = 0;
        }
        else{
			if(!TryEnterCriticalSection(&CriticalSection1))
                return false;

			int iBreakPackedResult = 1000;

            if(bcConnected[iClient] && !bcDisconnect[iClient])
            {
				if( (iBreakPackedResult = BreakClassPacket(iClient, outsBuf, outiSize)) == 4)
				{
					LeaveCriticalSection(&CriticalSection1);
					return false;
				}
            }
			else{
				LeaveCriticalSection(&CriticalSection1);
				return false;
			}

            LeaveCriticalSection(&CriticalSection1);

            if(*outiSize < 0 && !bcDropBadPackets[iClient])
            {
                FatalError(__FILE__, __LINE__, "GetClientData :: BreakClassPacket Failed.");
                bcDisconnect[iClient] = true;
                *outiSize = 0;
                return false;
            }
            else if(*outiSize < 0 && bcDropBadPackets[iClient]){
                icRecvBufSz[iClient] = 0;
                *outiSize = 0;
                return false;
            }

            if(iBreakPackedResult == 3)
			{
                icRecvBufSz[iClient] = 0;
			}
        }
        return true;
    }
    return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is called for each connecting client just
	before the client's bcConnected flag is set to TRUE.

	If this function returns FALSE the client wll not be accepted or connected.
*/
bool CSockSrvr::OnAcceptConnectClient(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::OnAcceptConnectClient.\n");
    #endif

	if(!Events.OnAcceptConnect(this, iClient))
	{
		return false;
	}
	
	//Allocate RAM for the clients receive buffer.
	if((scRecvBuf[iClient] = (char *) calloc(sizeof(char), INITSENDRECVSZ + 1)) == NULL)
    {
        return false;
    }

	//Allocate RAM for the clients send buffer.
    if((scSendBuf[iClient] = (char *) calloc(sizeof(char), INITSENDRECVSZ + 1)) == NULL)
    {
        return false;
    }

	icMaxSendBufSz[iClient] = INITSENDRECVSZ;
	icMaxRecvBufSz[iClient] = INITSENDRECVSZ;
    icRecvOffset[iClient] = 0;
    icRecvBufSz[iClient] = 0;
    icSendBufSz[iClient] = 0;
    bcRawDataMode[iClient] = false;

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is called for each disconnecting client just
	after the client's bcConnected flag is set to FALSE.

	NOTE: This function should not be called outside of the DisconnectClient function.
*/
void CSockSrvr::OnDisconnectClient(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::OnDisconnectClient.\n");
    #endif

	Events.OnDisconnect(this, iClient);
	
	free(scRecvBuf[iClient]);
    free(scSendBuf[iClient]);

    icRecvBufSz[iClient] = 0;
    icSendBufSz[iClient] = 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is called to disconnect a client.

	This functions return value has no meaning.

	NOTE: This function should not be called outside of the TCPHandler_Thread_Function.
*/
bool CSockSrvr::DisconnectClient(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::DisconnectClient.\n");
    #endif

    //The client is not connected.
	if(bcConnected[iClient] == false)
	{
		bcDisconnect[iClient] = false;
		return true;
	}
	
	if(!TryEnterCriticalSection(&CriticalSection1))
        return false;

    bcConnected[iClient] = false;
    bcDisconnect[iClient] = false;

    WaitOnClientThreadToExit(iClient);

    icCurrentClients--;

    WSASendDisconnect(sckcSocket[iClient], NULL);
	shutdown(sckcSocket[iClient], SD_BOTH);
	closesocket(sckcSocket[iClient]);

    OnDisconnectClient(iClient);

    LeaveCriticalSection(&CriticalSection1);

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function searched the bcConnected[] array to find a free client slot.

	If this function returns -1 the client wll not be accepted or connected.
*/
int CSockSrvr::GetFreeSocketSlot(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::GetFreeSocketSlot.\n");
    #endif

	int iClient = 0;

	while(iClient < icMaxClients)
	{
		if( bcConnected[iClient] == false && bcDisconnect[iClient] == false)
		{
            return iClient;
		}

		iClient++;
	}

	return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Initialize all of the class variables.
*/
bool CSockSrvr::Initialize(int iMaxClients)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::Initialize.\n");
    #endif

	int iClient = 0;

	if(!Start_WindowsSocket())
    {
        FatalError(__FILE__, __LINE__, "Start_WindowsSocket :: Failed.");
        return false;
	}

    icMaxClients = iMaxClients;
    icCurrentClients = 0;
    icNextClientID = 0;

	//Allocate memory for all of our class variables.
    if((bcWasConnect = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
        return false;

	if((icMaxSendBufSz = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
        return false;

	if((icMaxRecvBufSz = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
        return false;

	if((icRecvBufSz = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
        return false;

	if((scSendBuf = (char **) calloc(sizeof(char *), iMaxClients + 1)) == NULL)
        return false;

	if((scRecvBuf = (char **) calloc(sizeof(char *), iMaxClients + 1)) == NULL)
        return false;

	if((icSendBufSz = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
        return false;
    
	if((bcConnected = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
        return false;
    
	if((bcDisconnect = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
        return false;
    
	if((sckcSocket = (SOCKET *) calloc(sizeof(SOCKET), iMaxClients + 1)) == NULL)
        return false;
    
	if((icClientID = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
        return false;
    
	if((dwcClient_Thread_ID = (DWORD *) calloc(sizeof(DWORD), iMaxClients + 1)) == NULL)
        return false;
    
	if((hcClient_Thread_Handle = (HANDLE *) calloc(sizeof(HANDLE), iMaxClients + 1)) == NULL)
        return false;
    
	if((icRecvOffset = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
        return false;
    
	if((bcRawDataMode = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
        return false;
    
	if((bcDropBadPackets = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
        return false;

	//Initialize our Critical Section object.
	InitializeCriticalSection(&CriticalSection1);

    //Set all of our Client variables to their default values.
    while(iClient < iMaxClients)
	{
        bcConnected[iClient] = false;
        bcDisconnect[iClient] = false;
		bcDropBadPackets[iClient] = false;
		bcRawDataMode[iClient] = false;
		bcWasConnect[iClient] = false;
		scRecvBuf[iClient] = NULL;
		scSendBuf[iClient] = NULL;
		icClientID[iClient] = 0;
		icMaxSendBufSz[iClient] = 0;
		icMaxRecvBufSz[iClient] = 0;
		icRecvOffset[iClient] = 0;
		icRecvBufSz[iClient] = 0;
		icSendBufSz[iClient] = 0;
		sckcSocket[iClient] = NULL;
		iClient++;
	}

    IsListeningServer = false;

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Destroy all of the class variables.
*/
bool CSockSrvr::UnInitialize(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::UnInitialize.\n");
    #endif

    free(bcWasConnect);
	free(icMaxSendBufSz);
	free(icMaxRecvBufSz);
	free(scSendBuf);
	free(scRecvBuf);
	free(icSendBufSz);
	free(icRecvBufSz);
	free(bcConnected);
	free(bcDisconnect);
	free(sckcSocket);
	free(icClientID);
	free(dwcClient_Thread_ID);
	free(hcClient_Thread_Handle);
	free(icRecvOffset);
	free(bcRawDataMode);
	free(bcDropBadPackets);

    //Delete our Critical Section object.
	DeleteCriticalSection(&CriticalSection1);

	if(!Stop_WindowsSocket())
    {
        FatalError(__FILE__, __LINE__, "Stop_WindowsSocket :: Failed.");
        return false;
	}

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function sets up a socket to listen for incomming connections.
	This function should not be called outside of the Start() function.
*/
bool CSockSrvr::Open_ListenSocket(int iListenPort)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::Open_ListenSocket.\n");
    #endif

	SOCKADDR_IN LocalListenSocketAddress; // Socket address of local server

	// Fill In The Address Structure For Local Server
	LocalListenSocketAddress.sin_family      = AF_INET;		       // Address Family.
	LocalListenSocketAddress.sin_addr.s_addr = INADDR_ANY;         // Let socket Library Assign Address.
	LocalListenSocketAddress.sin_port        = htons(iListenPort); // Port Number.

	// Create A TCP/IP Stream Socket To "Listen" with.
	if((sckcListenSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == INVALID_SOCKET)
	{
        FatalError(__FILE__, __LINE__, "socket failed.");
		return false;
	}

	// Bind The Name To The Socket.
	if(bind(sckcListenSocket, (SOCKADDR*)&LocalListenSocketAddress, sizeof(struct sockaddr)) == SOCKET_ERROR)
	{
        FatalError(__FILE__, __LINE__, "bind failed.");
		closesocket(sckcListenSocket);
		return false;
	}

	// Set The Socket To Listen.
	if(listen(sckcListenSocket, SOMAXCONN) == SOCKET_ERROR)
	{
        FatalError(__FILE__, __LINE__, "listen failed.");
		closesocket(sckcListenSocket);
		return false;
	}

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function closes down the listening socket.
*/
bool CSockSrvr::Close_ListenSocket(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::Close_ListenSocket.\n");
    #endif

    if(closesocket(sckcListenSocket) == SOCKET_ERROR)
    {
        return false;
    }

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Start the client or server.

	If the iListenPort is non-zero a listening port will be opened to
	accept incomming connections.
*/
bool CSockSrvr::Start(int iListenPort)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::Start.\n");
    #endif

	if(!Events.OnStart(this, iListenPort))
	{
		return false;
	}
	
	//Are we going to be listening on any port?
	if(iListenPort > 0)
    {
        //Open our listening port.
		if(!Open_ListenSocket(iListenPort))
        {
            FatalError(__FILE__, __LINE__, "Open_ListenSocket :: Failed.");
            return false;
        }
        IsListeningServer = true;
    }

    if(!Start_TCPHandler())
    {
        FatalError(__FILE__, __LINE__, "Start_TCPHandler :: Failed.");
        return false;
    }

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Stop the server or client. If there is a listening socket
	it will be shutdown and closed.
*/
bool CSockSrvr::Stop(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::Stop.\n");
    #endif

	if(!Events.OnStop(this))
	{
		return false;
	}
	
	if(!Stop_TCPHandler())
    {
        FatalError(__FILE__, __LINE__, "Stop_TCPHandler :: Failed.");
        return false;
    }

    if(IsListeningServer)
    {
        if(!Close_ListenSocket())
        {
            FatalError(__FILE__, __LINE__, "Close_ListenSocket :: Failed.");
            return false;
        }
        IsListeningServer = false;
    }

    return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to actually send a clients data. It should never be called manually
	outside of DoSend or SendData.
*/
int CSockSrvr::SendDataEx(int iClient, char *sData, int iSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SendDataEx.\n");
    #endif

    int iRetVal = 0;

	if(!Events.OnSend(this, iClient, sData, iSize))
	{
		return 1; //All is well.
	}
	
	if(bcRawDataMode[iClient])
    {
    	//Just send the data.
		iRetVal = send(sckcSocket[iClient], sData, iSize, 0);
        return iRetVal;
    }
    else {
        char sSendData[IDEAL_SEND_SIZE];
        //Assemble a packet.
		int iSendLen = MakeClassPacket(sSendData, sData, iSize);
    	//Send the data.
		iRetVal = send(sckcSocket[iClient], sSendData, iSendLen, 0);
        return iRetVal;
    }

    FatalError(__FILE__, __LINE__, "SendDataEx :: Logic failure.");

    return -1;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to actually send a clients data. It should never be called manually.
*/
int CSockSrvr::SendData(int iClient, char *sData)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SendData.\n");
    #endif

	return SendDataEx(iClient, sData, strlen(sData));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CSockSrvr::ResizeSendBuffer(int iClient, int iSize)
{
	if((scSendBuf[iClient] = (char *) realloc(scSendBuf[iClient], iSize + 1)) == NULL)
	{
		return false;
	}
	else icMaxSendBufSz[iClient] = iSize + 1;
	return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CSockSrvr::ResizeRecvBuffer(int iClient, int iSize)
{
	if((scRecvBuf[iClient] = (char *) realloc(scRecvBuf[iClient], iSize + 1)) == NULL)
	{
		return false;
	}
	else icMaxRecvBufSz[iClient] = iSize + 1;
	return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the CriticalSection was not free,
	the client has data waiting to be sent, or if the client
	has been disconnected or is pending a disconnect.
*/
bool CSockSrvr::SetSendDataEx(int iClient, char *sData, int iSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SetSendDataEx.\n");
    #endif

    //Do we alread have data ready to be sent?
	if(!IsClientDataPending(iClient))
    {
        if(!TryEnterCriticalSection(&CriticalSection1))
            return false;

        //Is the client is connected and not ready to be disconnected?
		if(bcConnected[iClient] && !bcDisconnect[iClient])
        {
            //Is our clients send buffer large enough to handle the outgoing data?
			if(iSize > icMaxSendBufSz[iClient])
			{
				//Make our send buffer large enough to handle the outgoing data.
				if(!ResizeSendBuffer(iClient, iSize))
				{
					bcDisconnect[iClient] = true;
					LeaveCriticalSection(&CriticalSection1);
					return false;
				}
			}

			//Copy the outgoing data to our send buffer.
			memcpy(scSendBuf[iClient], sData, iSize);
			icSendBufSz[iClient] = iSize;
        }
		else{
	        LeaveCriticalSection(&CriticalSection1);
			return false;
		}

        LeaveCriticalSection(&CriticalSection1);
        return true;
    }

	return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the CriticalSection was not free,
	the client has data waiting to be sent, or if the client
	has been disconnected or is pending a disconnect.
*/
bool CSockSrvr::SetSendData(int iClient, char *sData)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SetSendData.\n");
    #endif

	return SetSendDataEx(iClient, sData, strlen(sData));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
*/
bool CSockSrvr::SetNextSendDataEx(int iClient, char *sData, int iSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SetNextSendDataEx.\n");
    #endif

    bool bRetVal = false;

    while(!(bRetVal = SetSendDataEx(iClient, sData, iSize)) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

	return bRetVal;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
*/
bool CSockSrvr::SetNextSendData(int iClient, char *sData)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> CSockSrvr::SetNextSendData.\n");
    #endif

	return SetNextSendDataEx(iClient, sData, strlen(sData));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif

